<?php
$config   = new \Config\App();
$appName  = $config->appName . " - Create Calendar Plan";
include('header.php');
?>

<main>
  <div class="container-fluid">
    <!-- Breadcrumb -->
    <div class="row m-1">
      <div class="col-12">
        <h4 class="main-title">
          Create Calendar for <?= esc($company['name']) ?>
        </h4>
        <span>Create an AI-generated Calendar Plan.</span>
      </div>
    </div>

    <div class="row position-relative">
      <!-- Left: Calendar Plan table -->
      <div class="col-lg-8 col-xxl-8 box-col-7">
        <div class="card">
          <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0">Calendar Plan</h5>

            <!-- Always render the button (hidden by default) -->
            <button id="approve-all"
                    class="btn btn-sm btn-primary"
                    disabled
                    style="display: none">
              Approve All
            </button>
          </div>
          <div class="card-body">
            <div class="table-responsive">
              <table class="table table-bordered table-striped align-middle mb-0">
                <thead>
                  <tr>
                    <th>Date</th>
                    <th>Title</th>
                    <th>Description</th>
                  </tr>
                </thead>
                <tbody id="plan-rows">
                  <?php if (count($plans) > 0): ?>
                    <?php foreach ($plans as $p): ?>
                      <tr data-plan-id="<?= esc($p['id']) ?>"
                          data-status="<?= empty($p['status']) ? 'pending' : 'approved' ?>">
                        <td><?= date('d-m-Y', strtotime($p['date'])) ?></td>
                        <td><?= esc($p['title']) ?></td>
                        <td><?= esc($p['description']) ?></td>
                      </tr>
                    <?php endforeach; ?>
                  <?php else: ?>
                    <tr>
                      <td colspan="3" class="text-center">
                        No calendar plan yet. Generate one via chat.
                      </td>
                    </tr>
                  <?php endif; ?>
                </tbody>
              </table>
            </div>
          </div>
        </div>
      </div>

      <!-- Right: Chat UI -->
      <div class="col-lg-4 col-xxl-4 box-col-5">
        <div class="card chat-container-content-box">
          <div class="card-header">
            <div class="chat-header d-flex align-items-center">
              <div class="d-lg-none">
                <a class="me-3 toggle-btn" role="button">
                  <i class="ti ti-align-justified"></i>
                </a>
              </div>
              <span class="profileimg h-45 w-45 d-flex-center b-r-50 bg-light position-relative">
                <img src="<?= base_url('assets/assets/images/avtar/14.png') ?>"
                     class="img-fluid b-r-50" alt="">
                <span class="position-absolute top-0 end-0 p-1 bg-success border border-light rounded-circle"></span>
              </span>
              <div class="flex-grow-1 ps-2 pe-2">
                <div class="fs-6">You</div>
                <div class="text-muted f-s-12 text-success">Online</div>
              </div>
            </div>
          </div>

          <div class="card-body chat-body">
            <div class="chat-container" id="chat-container">
              <?php foreach ($chats as $c): ?>
                <div class="position-relative">
                  <?php if ($c['role'] === 'user'): ?>
                    <div class="chat-box-right">
                      <p class="chat-text"><?= esc($c['message']) ?></p>
                    </div>
                  <?php else: ?>
                    <div class="chat-box">
                      <p class="chat-text"><?= esc($c['message']) ?></p>
                    </div>
                  <?php endif; ?>
                </div>
              <?php endforeach; ?>
            </div>
          </div>

          <div class="card-footer">
            <!-- Show "Create Plan" only if no plans exist -->
            <?php if (count($plans) === 0): ?>
              <button id="generate-plan" class="btn btn-primary w-100 mb-2">
                Create Plan
              </button>
            <?php endif; ?>

            <!-- Chat input: hidden initially when no plans -->
            <div id="chat-input-container"
                 <?= count($plans) === 0 ? 'style="display:none"' : '' ?>>
              <form id="chat-form" class="d-flex">
                <input type="hidden" name="company_id" value="<?= esc($company['id']) ?>">
                <div class="input-group w-100">
                  <input id="msg-input"
                         name="message"
                         class="form-control"
                         placeholder="Type a message…"
                         autocomplete="off"
                         required>
                  <button type="submit" class="btn btn-light-primary ms-2">
                    Send
                  </button>
                </div>
              </form>
            </div>
          </div>
        </div>
      </div>
    </div>
  </div>
</main>

<script>
  const chatForm         = document.getElementById('chat-form');
  const chatContainer    = document.getElementById('chat-container');
  const planRows         = document.getElementById('plan-rows');
  const approveAllBtn    = document.getElementById('approve-all');
  const generatePlanBtn  = document.getElementById('generate-plan');
  const chatInputWrapper = document.getElementById('chat-input-container');
  let initialTrigger     = false;

  // If the "Create Plan" button exists, wire it up to fire your first chat submission
  if (generatePlanBtn) {
    chatInputWrapper.style.display = 'none';
    generatePlanBtn.addEventListener('click', () => {
      initialTrigger = true;
      generatePlanBtn.disabled = true;
      chatForm.dispatchEvent(new Event('submit', { bubbles: true, cancelable: true }));
      // Now show chat input for follow-ups
      chatInputWrapper.style.display = '';
      generatePlanBtn.remove();
    });
  }

  // Enable/disable & show/hide Approve All based on row count & status
  function rebuildApproveState() {
    if (!approveAllBtn) return;

    const rows = planRows.querySelectorAll('tr[data-plan-id]');
    if (rows.length === 0) {
      // No plans = hide button
      approveAllBtn.style.display = 'none';
      return;
    }

    // There are plans = show button
    approveAllBtn.style.display = '';

    // Enable if any pending remain
    const anyPending = Array.from(rows).some(r => r.dataset.status === 'pending');
    approveAllBtn.disabled = !anyPending;
    approveAllBtn.innerText = anyPending ? 'Approve All' : 'Approved';
  }

  // On sending a chat message (or initial trigger), call your controller
  chatForm.addEventListener('submit', async e => {
    e.preventDefault();
    const data = new FormData(chatForm);
    const msg  = (data.get('message')||'').trim();
    if (!msg && !initialTrigger) return;

    if (msg) {
      chatContainer.insertAdjacentHTML('beforeend', `
        <div class="position-relative">
          <div class="chat-box-right"><p class="chat-text">${msg}</p></div>
        </div>
      `);
      chatContainer.scrollTop = chatContainer.scrollHeight;
    }
    initialTrigger = false;

    // Show typing placeholder
    const ph = document.createElement('div');
    ph.id = 'typing-ph';
    ph.className = 'position-relative';
    ph.innerHTML = `
      <div class="chat-box placeholder-glow">
        <span class="placeholder col-6"></span>
      </div>`;
    chatContainer.appendChild(ph);
    chatContainer.scrollTop = chatContainer.scrollHeight;

    // Table placeholder
    planRows.innerHTML = `
      <tr>
        <td class="placeholder-glow"><span class="placeholder col-4"></span></td>
        <td class="placeholder-glow"><span class="placeholder col-6"></span></td>
        <td class="placeholder-glow"><span class="placeholder col-5"></span></td>
      </tr>`;

    // Fetch from your controller
    const res  = await fetch('<?= site_url('home/generateCalendarPlan') ?>', {
      method: 'POST',
      body: data
    });
    const json = await res.json();

    // Remove typing placeholder
    document.getElementById('typing-ph')?.remove();

    // Append AI reply
    chatContainer.insertAdjacentHTML('beforeend', `
      <div class="position-relative">
        <div class="chat-box"><p class="chat-text">${json.ai_reply}</p></div>
      </div>
    `);
    chatContainer.scrollTop = chatContainer.scrollHeight;

    // Populate the table
    planRows.innerHTML = '';
    json.calendar_plan.forEach(item => {
      planRows.insertAdjacentHTML('beforeend', `
        <tr data-plan-id="${item.id}" data-status="pending">
          <td>${item.date}</td>
          <td>${item.title}</td>
          <td>${item.description}</td>
        </tr>
      `);
    });

    rebuildApproveState();
    chatForm.reset();
  });

  // Wire up the Approve All button
  if (approveAllBtn) {
    approveAllBtn.addEventListener('click', async () => {
      const ids = Array.from(
        document.querySelectorAll('#plan-rows tr[data-status="pending"]')
      ).map(r => r.dataset.planId);

      const res  = await fetch('<?= site_url('home/approvePlan') ?>', {
        method: 'POST',
        headers: {'Content-Type':'application/json'},
        body: JSON.stringify({plan_ids: ids})
      });
      const json = await res.json();

      if (json.success) {
        document.querySelectorAll('#plan-rows tr[data-status="pending"]')
                .forEach(r => r.dataset.status = 'approved');
        rebuildApproveState();
      } else {
        alert(json.message || 'Failed to approve');
      }
    });
  }

  // Initialize on load
  rebuildApproveState();
</script>

<?php include('footer.php'); ?>
